#!/sbin/sh
# updatekernel.sh
# 2010-12-25 Erasmux

# Note: ui_print will not handle new lines in the message and furthermore, for some reason,
# it also will not handle messages of length > 80.
ui_print() {
    echo ui_print "$@" 1>&$UPDATE_CMD_PIPE;
    if [ -n "$@" ]; then
        echo ui_print 1>&$UPDATE_CMD_PIPE;
    fi
}
log() { echo "$@"; }
fatal() { ui_print "$@"; exit 1; }

log ""

ui_print ""
ui_print "  Smart Kernel Updater v1.0"
ui_print "" 

kernelfile=kernel
basedir=`dirname $0`
BB=$basedir/busybox
awk="$BB awk"
grep="$BB grep"
chmod="$BB chmod"
chown="$BB chown"
chgrp="$BB chgrp"
cpio="$BB cpio"
find="$BB find"
gzip="$BB gzip"
gunzip="$BB gunzip"
tar="$BB tar"
errors=0
warning=0

updatename=`echo $UPDATE_FILE | $awk '{ sub(/^.*\//,"",$0); sub(/.zip$/,"",$0); print }'`
kernelver=`echo $updatename | $awk 'BEGIN {RS="-"; ORS="-"}; NR<=2 {print; ORS=""}`
args=`echo $updatename | $awk 'BEGIN {RS="-"}; NR>2 {print}'`

ui_print "Installing $kernelver..."

log "Update kernel script from $UPDATE_FILE started at $basedir"
ui_print "Parsing parameters..."
flags=
for pp in $args; do
  if [ "$pp" == "nologo" ]; then
      nologo=1
      flags="$flags -nologo"
      continue
  fi
  if [ "$pp" == "logo" ]; then
      logo=1
      flags="$flags -logo"
      continue
  fi
  if [ "$pp" == "noatime" ]; then
      noatime=1
      flags="$flags -noatime"
      continue
  fi
  if [ "$pp" == "atime" ]; then
      atime=1
      flags="$flags -atime"
      continue
  fi
  if [ "$pp" == "notweak" ]; then
      notweak=1
      flags="$flags -notweak"
      continue
  fi
  # noflash flag is a hidden flag for debugging
  if [ "$pp" == "noflash" ]; then
      noflash=1
      flags="$flags -noflash"
      continue
  fi
  vv=`echo $pp | $awk '/^m[0-9]+$/ { print substr($0,2) }'`
  if [ "$vv" -gt 0 ]; then
      if [ -n "$minfreq" ]; then
          ui_print "ERROR: multiple min freq: -m$vv"
          errors=$((errors + 1))          
          continue
      fi
      minfreq=`$awk "{ if (\\\$1 == $vv) { print \\\$2; exit } }" $basedir/frequencies.min`
      if [ "$minfreq" -gt 0 ]; then
          ui_print "min frequency: $minfreq"
      else
          minfreq=
          ui_print "ERROR: invalid min freq: -m$vv"
          ui_print "Supported min freqs are:"
          ui_print "`$awk 'NR>1 { printf \" \" }; { printf \"%3d\", $1 }' $basedir/frequencies.min`"
          errors=$((errors + 1))
      fi
      continue
  fi
  vv=`echo $pp | $awk '/^M[0-9]+$/ { print substr($0,2) }'`
  if [ "$vv" -gt 0 ]; then
      if [ -n "$maxfreq" ]; then
          ui_print "ERROR: multiple max freq: -M$vv"
          errors=$((errors + 1))          
          continue
      fi
      maxfreq=`$awk "{ if (\\\$1 == $vv) { print \\\$2; exit } }" $basedir/frequencies.max`
      if [ "$maxfreq" -gt 0 ]; then
          ui_print "max frequency: $maxfreq"
      else
          maxfreq=
          ui_print "ERROR: invalid max freq: -M$vv"
          ui_print "Supported max freqs are:"
          ui_print "`$awk 'NR>1 { printf \" \" }; { printf \"%3d\", $1 }' $basedir/frequencies.max`"
          errors=$((errors + 1))
      fi
      continue
  fi
#  vv=$basedir/$pp.rc
#  if [ -d $vv ]; then
#      rcdir=$vv
#      ui_print "using ramdisk for $pp"
#      continue;
#  fi
  errors=$((errors + 1))
  ui_print "ERROR: unknown argument -$pp"
done

if [ -n "$flags" ]; then
    ui_print "flags:$flags"
fi

if [ -n "$logo" -a -n "$nologo" ]; then
    ui_print "ERROR: -logo and -nologo are mutually exclusive"
    errors=$((errors + 1))
fi

if [ -n "$notweak" ]; then
    if [ -n "$logo" -o -n "$nologo" -o -n "$minfreq" -o -n "$maxfreq" ]; then
        ui_print "ERROR: -notweak is mutually exclusive with all other options"
        errors=$((errors + 1))
    fi
fi

if [ $errors -gt 0 ]; then
    fatal "argument parsing failed, aborting."
fi

ui_print "Extracting current kernel..."
log "dumping previous kernel image to $basedir/${boot}.img"
dump_image boot $basedir/boot.old
log "unpacking kernel image"
ramdisk=$basedir/boot.old-ramdisk.gz
$basedir/unpackbootimg $basedir/boot.old $basedir/
if [ "$?" -ne 0 -o ! -f $ramdisk ]; then
    fatal "ERROR: unpacking old boot image failed"
fi

if [ -z "$notweak" ]; then
    log "unpacking kernel ramdisk"
    mkdir $basedir/ramdisk
    cd $basedir/ramdisk
    $gunzip -c ../boot.old-ramdisk.gz | $cpio -i
    if [ "$?" -ne 0 -o ! -f init.rc ]; then
        ui_print "WARNING: unpacking ramdisk failed"        
        set notweak=1
    elif [ ! -f init.hero.rc ]; then
        ui_print "WARNING: invalid kernel ramdisk?!"
        set notweak=1
    fi
    if [ -n "$notweak" ]; then
        ui_print "Skipping ramdisk tweaks..."
        warning=$((warning + 1))
    fi
else
    ui_print "Skipping tweaks..."
fi

if [ -z "$notweak" ]; then
    ui_print "Applying tweaks..."

    # tweak ramdisk:
    if [ -z "$logo" ]; then
        logo=0
    fi
    if [ -z "$nologo" ]; then
        nologo=0
    fi
    if [ -z "$atime" ]; then
        atime=0
    fi
    if [ -z "$noatime" ]; then
        noatime=0
    fi
    mv init.rc ../init.rc.org
    $awk -f $basedir/initrc_tweak.awk -v logo=$logo -v nologo=$nologo -v noatime=$noatime -v atime=$atime ../init.rc.org > init.rc

    if [ -n "$minfreq" -o -n "$maxfreq" ]; then
        mv init.hero.rc ../init.hero.rc.org
        $awk -f $basedir/initherorc_tweak.awk -v minfreq=$minfreq -v maxfreq=$maxfreq ../init.hero.rc.org > init.hero.rc

        if [ -z "`$grep $minfreq init.hero.rc`" -o -z "`$grep $minfreq init.hero.rc`" ]; then
            ui_print "WARNING: failed applying min/max frequencies."
            warning=$((warning + 1))
        fi
    fi

    # build new ramdisk:

    if [ -f init.rc -a -f init.hero.rc ]; then
        $find . | $cpio -o -H newc | $gzip > ../boot.img-ramdisk.gz
        if [ "$?" -ne 0 -o ! -f $basedir/boot.img-ramdisk.gz ]; then
            ui_print "WARNING: ramdisk repacking failed."
            warning=$((warning + 1))
        else
            ramdisk=$basedir/boot.img-ramdisk.gz
        fi
    else
        ui_print "WARNING: ramdisk tweaks failed."
        warning=$((warning + 1))
    fi

    # tweak init.d scripts: (actually "untweak" init.d scripts)

    if [ -d /system/etc/init.d ]; then
        cd /system/etc/init.d
        mkdir $basedir/init.d
        failed=
        for initfile in *; do
            $awk -f $basedir/initd_tweak.awk -v untweak_minfreq=$minfreq -v untweak_maxfreq=$maxfreq $initfile > $basedir/init.d/$initfile
            if [ ! -f $basedir/init.d/$initfile ]; then
                failed=1
            fi
        done
        if [ -n "$failed" ]; then
            ui_print "WARNING: init.d tweaks failed."
            warning=$((warning + 1))
            $BB rm -rf $basedir/init.d
        else
            cd $basedir
            $chown -R 0 init.d
            $chgrp -R 2000 init.d
            $chmod -R 750 init.d
            $chmod 755 init.d
        fi
        # notice we still haven't touched the files under /system/etc/init.d
        # we'll replace then and after we flash the kernel successfully.
    fi
fi

ui_print "Packing kernel..."
cd $basedir
$basedir/mkbootimg --kernel $kernelfile --ramdisk $ramdisk --base "`cat boot.old-base`" --cmdline "`cat boot.old-cmdline`" --output boot.img
if [ "$?" -ne 0 -o ! -f boot.img ]; then
    fatal "ERROR: packing kernel failed?!"
fi
if [ -n "$noflash" ]; then
    fatal "Stopping before flash, -noflash given."
fi
ui_print "Flashing new kernel..."
flash_image boot $basedir/boot.img
if [ "$?" -ne 0 ]; then
    fatal "ERROR: flashing kernel failed?!"
fi

ui_print "Installing kernel modules..."
cd /system
$BB rm -rf lib/modules
$tar xvf $basedir/syslibs.tar
if [ "$?" -ne 0 -o ! -d /system/lib/modules ]; then
        ui_print "WARNING: kernel modules not installed?!"
        warning=$((warning + 1))
fi

if [ -d $basedir/init.d ]; then
    ui_print "Applying init.d (un)tweaks..."
    $BB rm -rf /system/etc/init.d
    $BB cp -a $basedir/init.d /system/etc/init.d
    if [ ! -d /system/etc/init.d ]; then
        ui_print "WARNING: init.d applying failed."
        warning=$((warning + 1))
    fi
fi

ui_print ""
if [ $warning -gt 0 ]; then
    ui_print "$kernelver installed with $warning warnings."
else
    ui_print "$kernelver installed successfully, enjoy :)"
fi
