#!/bin/sh

MTD_PARTS="5 6 8 9"
VOL_NUMS="0 1 2 3"
VOL_NAMES="system data cache efs"
TEMP_MNT="/tmp/mnt"
BACKUP_PATH="/sdcard/samdroid"

# Prepare the environment
export LD_LIBRARY_PATH=/tmp/tools
export PATH=/tmp/tools:/bin
if [ ! -e /lib ]; then
	mkdir /lib;
fi

if [ -e /lib/ld-uClibc.so.0 ]; then
	rm /lib/ld-uClibc.so.0
fi
ln -s /tmp/tools/ld-uClibc.so.0 /lib/ld-uClibc.so.0

# Unmount all UBI volumes
for vol in `cat /proc/mounts | cut -f 1 -d " " | grep ubi`; do
	umount $vol
	if [ $? -ne 0 ]; then
		echo Failed to unmount $vol, busy?
		exit 1
	fi
	echo Unmounted $vol.
done

# Detach all UBI devices
for dev in `ls -1 /dev/ubi* | grep -v "_" | cut -c 9-`; do
	ubidetach /dev/ubi_ctrl -d $dev
	if [ $? -ne 0 ]; then
		echo Failed to detach device $dev, check kernel output.
		exit 1
	fi
	echo Detached UBI device $dev.
done

# Make a test mount point
if [ ! -e $TEMP_MNT ]; then
	mkdir $TEMP_MNT
fi
if [ $? -ne 0 ]; then
	echo Failed to create temporary mount point $TEMP_MNT.
	exit 1
fi

# Rebuild UBI volumes
for part in $MTD_PARTS; do
	NAME=`echo ${VOL_NAMES} | cut -f 1 -d " "`
	VOL_NAMES=`echo ${VOL_NAMES} | cut -f 2- -d " "`
	NUM=`echo ${VOL_NUMS} | cut -f 1 -d " "`
	VOL_NUMS=`echo ${VOL_NUMS} | cut -f 2- -d " "`

	# Erase it
	flash_erase /dev/mtd/mtd$part 0 0
	if [ $? -ne 0 ]; then
		echo Failed to erase MTD partition $part, check kernel output.
		exit 1
	fi
	echo Erased MTD partition $part.

	# Attach it
	ubiattach /dev/ubi_ctrl -m $part -d $NUM
	if [ $? -ne 0 ]; then
		echo Failed to attach MTD partition $part on UBI device $NUM, check kernel output.
		exit 1
	fi
	echo Attached MTD partition $part as UBI device $NUM.

	# Make a volume on it
	ubimkvol /dev/ubi$NUM -m -N $NAME
	if [ $? -ne 0 ]; then
		echo Failed to create UBI volume $NAME on device $NUM, check kernel output.
		exit 1
	fi
	echo Created UBI volume $NAME on device $NUM.

	# Mount 
	mount -t ubifs /dev/ubi${NUM}_0 $TEMP_MNT
	if [ $? -ne 0 ]; then
		echo Failed to mount UBI volume $NAME, check kernel output.
		exit 1
	fi
	echo Mounted UBI volume $NAME successfully.

	# Restore if possible
	if [ -e ${BACKUP_PATH}/${NAME}_restore.tar ]; then
		tar -xpf ${BACKUP_PATH}/${NAME}_restore.tar -C $TEMP_MNT
		if [ $? -ne 0 ]; then
			echo Failed to restore UBI volume $NAME contents from ${BACKUP_PATH}/${NAME}_restore.tar.
			exit 1
		fi
		echo Restored UBI volume $NAME contents from ${BACKUP_PATH}/${NAME}_restore.tar.
	fi

	# Sync
	sync

	# Unmount
	umount $TEMP_MNT
	echo Unmounted UBI volume $NAME.
done
