// Flash driver ported from flashruntimeupdate_am41dl6408g.i
#include "flash_api.h"

//
//	----------------------------- device table --------------------------------
// 

#define ALL_FLASH_MODELS

#define FLASHDRV_NONE 0
#define FLASHDRV_AM29 1
#define FLASHDRV_I28  2
#define FLASHDRV_ST	  3

static const FlashSctrCfg_t SctrCfgTbl[] =
{
#if defined(INTEL28F160C3T) || defined(ALL_FLASH_MODELS)
	{0,		0x008988C2},	// INTEL F160C3T
	{0,		FLASHDRV_NONE},
	{31,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(INTEL28F160C3B) || defined(ALL_FLASH_MODELS)
	{0,		0x008988C3},	// INTEL F160C3B
	{0,		FLASHDRV_NONE},
	{8,		8 * 1024},
	{31,	64 * 1024},
#endif
#if defined(INTEL28F320C3T) || defined(ALL_FLASH_MODELS)
	{0,		0x008988C4},	// INTEL F320C3T
	{0,		FLASHDRV_NONE},
	{63,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(INTEL28F320C3B) || defined(ALL_FLASH_MODELS)
	{0,		0x008988C5},	// INTEL F320C3B
	{0,		FLASHDRV_NONE},
	{8,		8 * 1024},
	{63,	64 * 1024},
#endif
#if defined(LRS1382SHARP) || defined(ALL_FLASH_MODELS)
	{0,		0x00B000B4},	// Sharp LRS1382
	{0,		FLASHDRV_NONE},
	{63,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(LRS1383SHARP) || defined(ALL_FLASH_MODELS)
	{0,		0x00B000B5},	// Sharp LRS1383
	{0,		FLASHDRV_NONE},
	{8,		8 * 1024},
	{63,	64 * 1024},
#endif
#if defined(LRS1392SHARP) || defined(ALL_FLASH_MODELS)
	{0,		0x00B000B0},	// Sharp LRS1392
	{0,		FLASHDRV_NONE},
	{127,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(LRS1896SHARPTB) || defined(ALL_FLASH_MODELS)
	{0,		0x00B00010},	// Sharp LRS1896TB
	{0,		FLASHDRV_I28},
	{255,	64 * 1024},
	{8,		8 * 1024},
	{8,		8 * 1024},
	{255,	64 * 1024},
#endif
#if defined(LRS1896SHARPBB) || defined(ALL_FLASH_MODELS)
	{0,		0x00B00011},	// Sharp LRS1896BB
	{0,		FLASHDRV_I28},
	{8,		8 * 1024},
	{255,	64 * 1024},
	{255,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(AM41DL3224GB) || defined(ALL_FLASH_MODELS)
	{0,		0x00012256},	// AMD AM41DL3224GB
	{0,		FLASHDRV_NONE},
	{8,		8 * 1024},
	{63,	64 * 1024},
#endif
#if defined(AM50DL128BG)  || defined(ALL_FLASH_MODELS)
	{0,		0x00017E02},	// AMD AM50DL128BG
	{0,		FLASHDRV_NONE},
	{8,		8 * 1024},
	{126,	64 * 1024},
	{8,		8 * 1024},
	{8,		8 * 1024},
	{126,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(AM41DL6408G)  || defined(ALL_FLASH_MODELS)
	{0,		0x00017E02},	// AMD AM41DL6408G
	{0,		FLASHDRV_NONE},
	{8,		8 * 1024},
	{126,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(AM49PDL640AG)  || defined(ALL_FLASH_MODELS)
	{0,		0x00017E15},	// AMD AM49PDL640AG
	{0,		FLASHDRV_NONE},
	{8,		8 * 1024},
	{126,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(AM49PDL129BH)  || defined(ALL_FLASH_MODELS)
	{0,		0x0001227E},	// AMD AM49PDL129BH
	{0,		FLASHDRV_NONE},
	{127,	64 * 1024},
	{16,	8 * 1024},
	{127,	64 * 1024},
#endif
#if defined(AT52BR3224) || defined(ALL_FLASH_MODELS)
	{0,		0x001F00C8},	// AT52BR3224
	{0,		FLASHDRV_NONE},
	{8,		8 * 1024},
	{63,	64 * 1024},
#endif
#if defined(AT52BR3224T) || defined(ALL_FLASH_MODELS)
	{0,		0x001F00C9},	// AT52BR3224T
	{0,		FLASHDRV_NONE},
	{63,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(AM29DL640G) || defined(ALL_FLASH_MODELS)
	{0,		0x0001007E},	// AM29DL640G
	{0,		FLASHDRV_AM29},
	{8,		8 * 1024},
	{126,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(BB_DEVICE_ID) || defined(ALL_FLASH_MODELS)
	{0,		0x00898855},	// BB_DEVICE_ID
	{0,		FLASHDRV_I28},
	{8,		8 * 1024},
	{127,	64 * 1024},
#endif
#if defined(TB_DEVICE_ID) || defined(ALL_FLASH_MODELS)
	{0,		0x00898854},	// TB_DEVICE_ID
	{0,		FLASHDRV_I28},
	{127,	64 * 1024},
	{8,		8 * 1024},
#endif
#if defined(BB_DEVICE2_ID) || defined(ALL_FLASH_MODELS)
	{0,		0x00898815},	// BB_DEVICE2_ID
	{0,		FLASHDRV_I28},
	{4,		32 * 1024},
	{127,	128 * 1024},
#endif
#if defined(TB_DEVICE2_ID) || defined(ALL_FLASH_MODELS)
	{0,		0x00898812},	// TB_DEVICE2_ID
	{0,		FLASHDRV_I28},
	{127,	128 * 1024},
	{4,		32 * 1024},
#endif
#if defined(ST_DEVICE) || defined(ALL_FLASH_MODELS)
	{0,		0x00208810},	// STM58WT064FT
	{0,		FLASHDRV_ST},
	{127,	64 * 1024},	//	32 kword
	{8,		8 * 1024},	//	4 kword
#endif
#if defined(ST_DEVICE) || defined(ALL_FLASH_MODELS)
	{0,		0x00208811},	// STM58WT064FB
	{0,		FLASHDRV_ST},
	{8,		8 * 1024},	//	4 kword
	{127,	64 * 1024},	//	32 kword
#endif
	{0,		0}				// End of the table
};

#define DQ7					0x80
#define DQ6					0x40
#define DQ5					0x20
#define DQ4					0x10
#define DQ3					0x08
#define DQ2					0x04
#define DQ1					0x02

#define TOGGLE_BITS			( DQ6 | DQ2 )

#define WRITE_CMD( addr, data )		( *((volatile UInt16 *)(addr)) = (UInt16)(data) )
#define	READ_WORD( addr ) 			( *((volatile UInt16 *)(addr)) )
#define WRITE_WORD( addr, data )	( *((volatile UInt16 *)(addr)) = (UInt16)(data) )

#define	READ_BYTE( addr ) 			( *((volatile UInt8 *)(addr)) )

#define COMMAND_RESET				0xf0	//write with XXX

#define WORD_ALIGNED( addr ) ( (UInt32)(addr)&0x1 ? 0:1 ) 

//
//	--	AM29 drivers
//

#define CommandSequenceADDR_1st		(0x555<<1)	// <<1 because addr A1 -> flash A0
#define CommandSequenceADDR_2nd		(0x2aa<<1)
#define CommandSequenceADDR_3rd		(0x555<<1)
#define CommandSequenceADDR_4th		(0x555<<1)
#define CommandSequenceADDR_5th		(0x2aa<<1)
#define CommandSequenceADDR_6th		(0x555<<1)

#define IdCheckData_1st	 	0xaa		// 1st data for Id check
#define IdCheckData_2nd	 	0x55		// 2nd data for Id check
#define IdCheckData_3rd	 	0x90		// 3rd data for Id check

static UInt32 am29_Identify( void )
{
	UInt8	mfg_id ;
	UInt8	dev_id ;

	WRITE_CMD( 0x0, COMMAND_RESET );

	WRITE_CMD( CommandSequenceADDR_1st, IdCheckData_1st );
	WRITE_CMD( CommandSequenceADDR_2nd, IdCheckData_2nd );
	WRITE_CMD( CommandSequenceADDR_3rd, IdCheckData_3rd );

	mfg_id = READ_WORD( 0x0 );
	dev_id = READ_WORD( 0x2 );

	WRITE_CMD( 0x0, COMMAND_RESET );

	return 	( mfg_id << 16 ) | dev_id ;
}

#define SectorErase_CycleData_1st 	0xaa	// 1st data for erase sector
#define SectorErase_CycleData_2nd 	0x55	// 2nd data for erase sector
#define SectorErase_CycleData_3rd 	0x80	// 3rd data for erase sector
#define SectorErase_CycleData_4th 	0xaa	// 4th data for erase sector
#define SectorErase_CycleData_5th 	0x55	// 5th data for erase sector
#define SectorErase_CycleData_6th 	0x30	// 6th data for erase sector

static void am29_StartErase( FlashSctrAddr_t sctrAddr )
{
	WRITE_CMD( sctrAddr, COMMAND_RESET );
	WRITE_CMD( sctrAddr, COMMAND_RESET );

	WRITE_CMD( CommandSequenceADDR_1st, SectorErase_CycleData_1st );
	WRITE_CMD( CommandSequenceADDR_2nd, SectorErase_CycleData_2nd );
	WRITE_CMD( CommandSequenceADDR_3rd, SectorErase_CycleData_3rd );
	WRITE_CMD( CommandSequenceADDR_4th, SectorErase_CycleData_4th );
	WRITE_CMD( CommandSequenceADDR_5th, SectorErase_CycleData_5th );

	WRITE_WORD( sctrAddr, SectorErase_CycleData_6th );
}

static Boolean am29_EraseDone( FlashSctrAddr_t sctrAddr )
{
	volatile UInt8 erase_status;
	volatile UInt8 DQ;

	DQ = READ_WORD(sctrAddr);
	erase_status = READ_WORD( sctrAddr );
	DQ ^= erase_status;

	if (!(DQ & (DQ6 | DQ2)))	// erase done
	{
		WRITE_CMD( 0x0, COMMAND_RESET );

		return TRUE ;
	}

	return FALSE ;
}

#define Program_CycleData_1st	 	0xaa		// 1st data for Program
#define Program_CycleData_2nd	 	0x55		// 2nd data for Program
#define Program_CycleData_3rd	 	0xa0		// 3rd data for Program

static void am29_StartWrite( FlashSctrAddr_t addr, UInt16 data )
{

	WRITE_CMD( addr, COMMAND_RESET );

	WRITE_CMD( CommandSequenceADDR_1st, Program_CycleData_1st );
	WRITE_CMD( CommandSequenceADDR_2nd, Program_CycleData_2nd );
	WRITE_CMD( CommandSequenceADDR_3rd, Program_CycleData_3rd );

	WRITE_WORD( addr, data );
}

static Boolean am29_WriteDone( FlashSctrAddr_t addr )
{
	volatile UInt8 DQ = (UInt8) READ_WORD(addr);
	volatile UInt8 status = READ_WORD(addr);

	DQ ^= status;
	if (!(DQ & DQ6))	// program done
	{
		WRITE_CMD( 0x0, COMMAND_RESET );
		return TRUE;
	}

	return FALSE ;
}

//
//	--	Intel28 drivers
//
#define COMMAND_BLOCK_ERASE			0x20
#define COMMAND_CLEAR_STATUS		0x50
#define COMMAND_PROGRAM				0x40
#define COMMAND_BUF_PROGRAM			0xE8
#define COMMAND_READ_ARRAY			0xFF
#define COMMAND_READ_CONFIG			0x90
#define COMMAND_READ_STATUS			0x70
#define	COMMAND_UNLOCK_BLOCK		0x60
#define DATA_ERASE_CONFIRM			0xD0
#define	DATA_UNLOCK_CONFIRM			0xD0
#define	COMMAND_BUF_PROGRAM_CONFIRM	0xD0

#define I28_RESET( addr ) 	WRITE_CMD( addr, COMMAND_READ_ARRAY );

static UInt32 i28_Identify( void )
{
	UInt16	mfg_id ;
	UInt16	dev_id ;
	UInt16 	i;

	I28_RESET( 0x0 );

	WRITE_CMD( 0x0, COMMAND_READ_CONFIG );
	
	for( i=0; i<500; i++ ); //wait for 150ns

	mfg_id = READ_WORD( 0 ) ;
	dev_id = READ_WORD( 2 ) ;

	I28_RESET( 0x0 );

	return 	( mfg_id << 16 ) | dev_id ;
}


static void i28_StartErase( FlashSctrAddr_t sctrAddr )
{
	UInt32 addr = (UInt32) sctrAddr ;

	I28_RESET( addr ) ;

	WRITE_CMD( addr, COMMAND_UNLOCK_BLOCK );
	WRITE_WORD( addr, DATA_UNLOCK_CONFIRM );
	WRITE_CMD( addr, COMMAND_CLEAR_STATUS );
	WRITE_CMD( addr, COMMAND_BLOCK_ERASE );
	WRITE_WORD( addr,DATA_ERASE_CONFIRM );
}

static Boolean i28_EraseDone( FlashSctrAddr_t sctrAddr )
{
	UInt8 status;

	WRITE_CMD( sctrAddr, COMMAND_READ_STATUS );

	status = READ_WORD( sctrAddr );

	if( status & DQ7 ) {
		WRITE_CMD( sctrAddr, COMMAND_READ_ARRAY );
		return TRUE ;
	}

	return FALSE ;
}

static void i28_StartWrite ( FlashSctrAddr_t addr, UInt16 data ) 
{
	I28_RESET( addr ) ;

	WRITE_CMD( addr, COMMAND_CLEAR_STATUS );
	WRITE_CMD( addr, COMMAND_PROGRAM );
	WRITE_WORD( addr, data );
}

static Boolean i28_WriteDone( FlashSctrAddr_t sctrAddr )
{
	UInt8 status;

	WRITE_CMD( sctrAddr, COMMAND_READ_STATUS );

	status = READ_WORD( sctrAddr );

	if( status & DQ7 ) {
		WRITE_CMD( sctrAddr, COMMAND_READ_ARRAY );
		return TRUE ;
	}

	return FALSE ;
}

static void i28_StartBufWrite ( FlashSctrAddr_t addr, UInt16 *data, UInt16 nWords ) 
{
	int i;

	I28_RESET( addr ) ;

//	UnlockBlock( (UInt32)addr );

	WRITE_CMD( addr, COMMAND_CLEAR_STATUS );
	WRITE_CMD( addr, COMMAND_BUF_PROGRAM );

	WRITE_WORD( addr, nWords-1 );

	for( i=0 ; i<nWords ; ++i ) {
		WRITE_WORD( addr+i, data[i] );
	}

	WRITE_WORD(addr, COMMAND_BUF_PROGRAM_CONFIRM);
}

static Boolean i28_BufWriteDone( FlashSctrAddr_t addr )
{
	UInt8 status;

	WRITE_CMD( addr, COMMAND_READ_STATUS );

	status = READ_WORD( addr );

	if( status & DQ7 ) {
			WRITE_CMD( addr, COMMAND_READ_ARRAY );
			return TRUE ;
	}

	return FALSE ;
}

//
//	------------------------ Get maximum sector size --------------------------
//

static UInt32 MaxSctrBytes( const FlashSctrCfg_t* sctrCfg )
{
	UInt32 maxSctrBytes = 0 ;

	do {
		if( sctrCfg->sctrBytes > maxSctrBytes ) {
			maxSctrBytes = sctrCfg->sctrBytes ;
		}
	} while( (++sctrCfg)->sctrCount ) ;

	return maxSctrBytes ;
}

//
//	-------------------------- Get total flash size ---------------------------
//

static UInt32 TotalBytes( const FlashSctrCfg_t* sctrCfg )
{
	UInt32 totalBytes = 0 ;

	do {
		totalBytes += sctrCfg->sctrBytes * sctrCfg->sctrCount ;
	} while( (++sctrCfg)->sctrCount ) ;

	return totalBytes ;
}

//
//	--------------------------- Find sector from address ----------------------
//

Result_t Flash_FindSector( FlashDev_t * dev, void * addr, FlashSctrAddr_t *sctrAddr, UInt32 *sctrBytes )
{
	const FlashSctrCfg_t *sctrCfg = dev->sctrCfg ;

	UInt32	i ;

	*sctrAddr = 0 ;
	
	do {
		UInt32 totalBytes = sctrCfg->sctrCount * sctrCfg->sctrBytes ;

		if( (UInt32)(*sctrAddr) + totalBytes > (UInt32)addr ) {
			break ;
		}

		*sctrAddr += ( totalBytes >> 1 ) ;

	} while ( (++sctrCfg)->sctrCount ) ;

	if( !sctrCfg->sctrCount ) {
		return RESULT_TOOL_FLASHDRV_ADDRESS_ERROR ;
	}

	for( i=0; i<sctrCfg->sctrCount; i++ ) {

		if( (UInt32)(*sctrAddr) + sctrCfg->sctrBytes > (UInt32)addr ) {
			break ;
		}

		*sctrAddr += ( sctrCfg->sctrBytes >> 1 ) ;
	}

	if( i==sctrCfg->sctrCount ) {
		return RESULT_TOOL_FLASHDRV_ADDRESS_ERROR ;
	}

	*sctrBytes = sctrCfg->sctrBytes ;

	return RESULT_OK ;
}

//
//	------------------------------- 'open' device -----------------------------
//
static Boolean GetDrivers( UInt32 flashDrv, FlashDev_t * dev )
{
	dev->Identify		= 0 ;
	dev->StartErase		= 0 ;
	dev->EraseDone		= 0 ;
	dev->StartWrite		= 0 ;
	dev->WriteDone		= 0 ;
	dev->StartBufWrite	= 0 ;
	dev->BufWriteDone	= 0 ;
	dev->bufWords		= 0 ;

	switch( flashDrv ) {

	case FLASHDRV_AM29:
		dev->Identify   = am29_Identify ;
		dev->StartErase = am29_StartErase ;
		dev->EraseDone  = am29_EraseDone ;
		dev->StartWrite = am29_StartWrite ;
		dev->WriteDone  = am29_WriteDone ;
		return TRUE ;
	
	case FLASHDRV_I28:
		dev->Identify		= i28_Identify ;
		dev->StartErase		= i28_StartErase ;
		dev->EraseDone		= i28_EraseDone ;
		dev->StartBufWrite	= i28_StartBufWrite ;
		dev->BufWriteDone	= i28_BufWriteDone ;
		dev->bufWords		= 16 ;					// the Intel TYAX flash supports 32-word buffered
													// programming; the SHARP flash (which uses the
													// Intel drivers) supports 16 words.  Measured timing
													// at 460k baudrate shows no difference between
													// 16 and 32 word buffered program on Intel flash --
													// so we'll use 16 to stay compatible with Sharp.
													// We may want to revisit this (for Intel) when
													// using USB vs. UART.
		return TRUE ;

	case FLASHDRV_ST:		//	use the i28 drivers -- they are compatible
		dev->Identify   = i28_Identify ;
		dev->StartErase = i28_StartErase ;
		dev->EraseDone  = i28_EraseDone ;
		dev->StartWrite = i28_StartWrite ;
		dev->WriteDone  = i28_WriteDone ;
		return TRUE ;
	
	default:
		return FALSE ;
	}
}

Result_t Flash_Open( FlashDev_t * dev )
{
	const FlashSctrCfg_t* sctrCfg = SctrCfgTbl ;
	UInt32 chipType ;
	UInt32 flashDrv ;

	dev->chipType = 0 ;
	dev->sctrCfg = 0 ;
	dev->maxSctrBytes = 0 ;
	dev->totalBytes = 0 ;

	while( 0 != ( chipType = sctrCfg++->sctrBytes ) ) {
	
		flashDrv = sctrCfg++->sctrBytes ;
	
		if( FLASHDRV_NONE != flashDrv ) {
			
			if( !GetDrivers( flashDrv, dev ) ) {
				return RESULT_TOOL_FLASHDRV_DEVICE_UNKNOWN ;
			}

			dev->chipType = dev->Identify( ) ;

			if( !dev->chipType ) {
				return RESULT_TOOL_FLASHDRV_DEVICE_UNKNOWN ;
			}
			
			if( dev->chipType == chipType ) {
				dev->sctrCfg = sctrCfg ;
				dev->maxSctrBytes = MaxSctrBytes( dev->sctrCfg ) ;
				dev->totalBytes = TotalBytes( dev->sctrCfg ) ;
				return RESULT_OK ;
			}
		}

		while( sctrCfg->sctrCount ) ++sctrCfg ;
	}

	return RESULT_TOOL_FLASHDRV_DEVICE_UNKNOWN ;
}

Result_t Flash_WriteSectorWithCallback( 
	FlashDev_t		*dev, 
	FlashSctrAddr_t	pgmAddr, 
	FlashSctrAddr_t	dataBuf,
	UInt32			dataBytes, 
	FlashCbFct_t	cbFunc
	 )
{
	FlashSctrAddr_t	sctrAddr ;
	UInt32			sctrBytes ;
	UInt32			i ;
	Result_t		result ;
	
	if( !WORD_ALIGNED( pgmAddr ) ) {
		return RESULT_TOOL_FLASHDRV_MEMALIGN_ERROR ;
	}

	if( !WORD_ALIGNED( dataBuf ) ) {
		return RESULT_TOOL_FLASHDRV_MEMALIGN_ERROR ;
	}

	if( dataBytes %2 ) ++dataBytes ;

	result= Flash_FindSector( dev, pgmAddr, &sctrAddr, &sctrBytes ) ;

	if( RESULT_OK != result ) {
		return result ;
	}

	if( ( (UInt32)pgmAddr - (UInt32)sctrAddr + dataBytes ) > sctrBytes ) {
		return RESULT_TOOL_FLASHDRV_ADDRESS_ERROR ;
	}

	dev->StartErase( sctrAddr ) ;
	
	do {
		if( cbFunc ) {
			cbFunc( ) ;
		}
	} while( !dev->EraseDone( sctrAddr ) ) ;

	for( i=0; i<(dataBytes>>1); i++ ) {

		dev->StartWrite( pgmAddr+i, dataBuf[i] ) ;

		do {
			if( cbFunc ) {
				cbFunc( ) ;
			}
		} while( !dev->WriteDone( sctrAddr ) ) ;
	}

	return RESULT_OK ;
}

Result_t Flash_WriteSector( 
	FlashDev_t		*dev, 
	FlashSctrAddr_t	pgmAddr, 
	FlashSctrAddr_t	dataBuf,
	UInt32			dataBytes
	 )
{
	return Flash_WriteSectorWithCallback( dev, pgmAddr, dataBuf, dataBytes, 0 ) ;
}



//
//	------------------------------ program a sector ---------------------------
//
Result_t Flash_UpdateSector( 
	FlashDev_t		*dev, 
	void			*pgmAddr, 
	void			*dataBuf,
	UInt32			 dataBytes,
	void			*saveBuf,
	UInt32			 saveBytes
	 )
{
	FlashSctrAddr_t	sctrAddr ;
	UInt32			sctrBytes ;
	UInt32			offsetBytes ;
	Result_t		result ;
	UInt32			i ;
	
	FlashSctrAddr_t	pSrc ;
	FlashSctrAddr_t	pDst ;

	UInt8			*pSrc8 ;
	UInt8			*pDst8 ;

	if( !saveBuf || !saveBytes ) {
		return Flash_WriteSector( dev, pgmAddr, dataBuf, dataBytes ) ;
	}

	result= Flash_FindSector( dev, pgmAddr, &sctrAddr, &sctrBytes ) ;

	if( RESULT_OK != result ) {
		return result ;
	}

	offsetBytes = (UInt32)pgmAddr - (UInt32)sctrAddr ;

	if( offsetBytes + dataBytes > sctrBytes ) {
		return RESULT_TOOL_FLASHDRV_ADDRESS_ERROR ;
	}

	pSrc = sctrAddr ;
	pDst = saveBuf ;

	for( i=0; i<(sctrBytes>>1); i++ ) {
		*pDst++ = *pSrc++ ;
	}

	pSrc8 = dataBuf ;
	pDst8 = (UInt8*)saveBuf + offsetBytes ;

	for( i=0; i<dataBytes; i++ ) {
		*pDst8++ = *pSrc8++ ;
	}

	return Flash_WriteSector( dev, sctrAddr, saveBuf, sctrBytes ) ;
}

Result_t Flash_ClearSector( 
	FlashDev_t		*dev, 
	void			*pgmAddr, 
	UInt32			 clearBytes,
	void			*saveBuf,
	UInt32			 saveBytes
	 ) 
{
	FlashSctrAddr_t	sctrAddr ;
	UInt32			sctrBytes ;
	UInt32			offsetBytes ;
	Result_t		result ;
	UInt32			i ;
	
	FlashSctrAddr_t	pSrc ;
	FlashSctrAddr_t	pDst ;

	UInt8			*pDst8 ;

	result= Flash_FindSector( dev, pgmAddr, &sctrAddr, &sctrBytes ) ;

	if( RESULT_OK != result ) {
		return result ;
	}

	offsetBytes = (UInt32)pgmAddr - (UInt32)sctrAddr ;

	if( offsetBytes + clearBytes > sctrBytes ) {
		return RESULT_TOOL_FLASHDRV_ADDRESS_ERROR ;
	}

	pSrc = sctrAddr ;
	pDst = saveBuf ;

	for( i=0; i<(sctrBytes>>1); i++ ) {
		*pDst++ = *pSrc++ ;
	}

	pDst8 = (UInt8*)saveBuf + offsetBytes ;

	for( i=0; i<clearBytes; i++ ) {
		*pDst8++ = 0 ;
	}

	return Flash_WriteSector( dev, sctrAddr, saveBuf, sctrBytes ) ;
}


Result_t Flash_ReadSector( 
	FlashDev_t		*dev, 
	void			*pgmAddr, 
	void			*dataBuf,
	UInt32			 dataBytes
	 ) 
{
	FlashSctrAddr_t	sctrAddr ;
	UInt32			sctrBytes ;
	UInt32			offsetBytes ;
	Result_t		result ;
	UInt32			i ;
	
	UInt8			*pSrc8 ;
	UInt8			*pDst8 ;

	result= Flash_FindSector( dev, pgmAddr, &sctrAddr, &sctrBytes ) ;

	if( RESULT_OK != result ) {
		return result ;
	}

	offsetBytes = (UInt32)pgmAddr - (UInt32)sctrAddr ;

	if( offsetBytes + dataBytes > sctrBytes ) {
		return RESULT_TOOL_FLASHDRV_ADDRESS_ERROR ;
	}

	pSrc8 = (UInt8*)sctrAddr + offsetBytes ;
	pDst8 = dataBuf ;

	for( i=0; i<dataBytes; i++ ) {
		*pDst8++ = *pSrc8++ ;
	}

	return RESULT_OK ;
}
