/**	@file flash_api.h &copy; 2004 Broadcom Corporation.  Defines
 *	the flash tool API
 */

#ifndef __FLASHAPI_H__
#define __FLASHAPI_H__

#include "types.h"
#include "consts.h"
#include "resultcode.h"

/**	Maximum sector size in bytes.  Based on the device table (SctrCfgTbl) defined in flash_api.c.
 *	can be used to allocate 'saveBuf' buffers in the Flash_Update() and Flash_ClearSector()
 *	API functions.
 */
#define FLASH_MAX_SECTOR_SIZE_BYTES ( 4 * 64 * 1024 )	// id=0x00898812, 0x00898815
#define MAX_SPARE_SIZE				(64*4*16)

#ifndef _WIN32

/**	Flash sector configuration.  This is used to build the device table (SctrCfgTbl) defined
 *	in flash_api.c.  It is based on the momo flash driver implementation.
 */
typedef struct
{
	UInt32					sctrCount;			/**<	number of sectors */
	UInt32					sctrBytes;			/**<	number of bytes per sector */
}	FlashSctrCfg_t;

/**	Default device memory pointer type
 */
typedef UInt16				*FlashSctrAddr_t ;

/**	Flash device descriptor.
 */
typedef UInt32  (*FlashIdentDrv_t)      ( void ) ;
typedef void    (*FlashStartEraseDrv_t) ( FlashSctrAddr_t sctrAddr ) ;
typedef Boolean (*FlashEraseDoneDrv_t)  ( FlashSctrAddr_t sctrAddr ) ;
typedef void    (*FlashStartWriteDrv_t) ( FlashSctrAddr_t, UInt16 ) ;
typedef Boolean (*FlashWriteDoneDrv_t)  ( FlashSctrAddr_t ) ;
typedef void	(*FlashStartBufWriteDrv_t)	( FlashSctrAddr_t, UInt16*, UInt16 ) ;
typedef Boolean	(*FlashBufWriteDoneDrv_t)	( FlashSctrAddr_t ) ;
typedef void    (*FlashCbFct_t)         ( void ) ;

typedef struct {
	UInt32					 chipType ;			/**<	'chip type' based on device query */
	const FlashSctrCfg_t	*sctrCfg ;			/**<	pointer to first sector configuration entry for this device */
	UInt32					 maxSctrBytes ;		/**<	max sector size for this device */
	UInt32					 totalBytes ;		/**<	total device size in bytes */
	FlashIdentDrv_t			 Identify ;			/**<	id chip driver */
	FlashStartEraseDrv_t	 StartErase ;		/**<	start-erase-sector driver */
	FlashEraseDoneDrv_t		 EraseDone ;		/**<	check-erase-sector done driver */
	FlashStartWriteDrv_t	 StartWrite ;		/**<	start-word-write driver */
	FlashWriteDoneDrv_t		 WriteDone ;		/**<	check-write done driver */
	FlashStartBufWriteDrv_t	 StartBufWrite ;	/**<	start-buffered-write driver */
	FlashBufWriteDoneDrv_t	 BufWriteDone ;		/**<	check-buffered-write done driver */
	UInt16					 bufWords ;			/**<	number of words buffered if buffered write supported */
}	FlashDev_t ;

/**	Open the flash device.  Queries the device for 'chip type' then searches SctrCfgTbl for a matching type.
 *	Also assigns the 'maxSctyrBytes' and 'totalBytes' fields of the FlashDev_t structure.
 */
Result_t Flash_Open( 
	FlashDev_t		*dev						/**<	the device */
	) ;

/**	Find sector based on address.  Returns the sector address and number of bytes in sector */
Result_t Flash_FindSector( 
	FlashDev_t		*dev, 						/**<	the device */
	void			*addr,						/**<	the address */
	FlashSctrAddr_t	*sctrAddr,					/**<	sector address returned */
	UInt32			*sctrBytes ) ;				/**<	number of bytes in sector returned */

/**	Write to a sector.  The sector is first erased.  Then 'dataBytes' bytes from 'dataBuf' are programmed
 *	to the sector starting at 'pgmAddr'.  The start and end address do not have to start/end on sector
 *	boundaries, but no sector boundary may be crossed.  \note Requires that pgmAddr, dataBuf and 
 *	dataBytes be FlashSctrAddr_t aligned.
 *	 
 */
Result_t Flash_WriteSector( 
	FlashDev_t		*dev, 						/**<	the device */
	FlashSctrAddr_t	 pgmAddr,					/**<	the program address */
	FlashSctrAddr_t	 dataBuf,					/**<	the program data */
	UInt32			 dataBytes					/**<	number of program data bytes */
	) ;

/**	 Same as Flash_WriteSector except that callback function 'cb' is called during the erase and
 *	 write wait cycles.  This can be used to display status or to overlap data fetches with
 *	 program operations.
 */
Result_t Flash_WriteSectorWithCallback( 
	FlashDev_t		*dev, 						/**<	the device */
	FlashSctrAddr_t	 pgmAddr,					/**<	the program address */
	FlashSctrAddr_t	 dataBuf,					/**<	the program data */
	UInt32			 dataBytes,					/**<	number of program data bytes */
	FlashCbFct_t	 cbFct						/**<	callback fn */
	 ) ;

/**	Update a sector.  The sector contents are read into the 'saveBuf' buffer, then the sector
 *	is erased.  The 'dataBytes' bytes from 'dataBuf' are copied to 'saveBuf' then the entire sector
 *	is programmed.  This results in data outside the specified program region to be preserved.
 *	The start and end address do not have to start/end on sector boundaries, but no sector boundary 
 *	may be crossed.  pgmAddr, dataBuf and dataBytes be aligned on even or odd byte boundaries,
 *	independent of FlashSctrAddr_t..
 *	 
 */
Result_t Flash_UpdateSector( 
	FlashDev_t		*dev, 						/**<	the device */
	void			*pgmAddr, 					/**<	the program address */
	void			*dataBuf,					/**<	the program data */
	UInt32			 dataBytes,					/**<	number of program data bytes */
	void			*saveBuf,					/**<	a save area at least the size of the sector */
	UInt32			 saveBytes					/**<	save area size in bytes */
	 ) ;

/**	Clear a sector.  The sector contents are read into the 'saveBuf' buffer, then the sector
 *	is erased.  The 'dataBytes' bytes in 'saveBuf' are cleared to zero, then the entire sector
 *	is programmed.  This results in data outside the specified program region to be preserved.
 *	The start and end address do not have to start/end on sector boundaries, but no sector boundary 
 *	may be crossed.  pgmAddr, dataBuf and dataBytes be aligned on even or odd byte boundaries,
 *	independent of FlashSctrAddr_t..
 *	 
 */
Result_t Flash_ClearSector( 
	FlashDev_t		*dev, 						/**<	the device */
	void			*pgmAddr,					/**<	the program address */
	UInt32			 clearBytes,				/**<	number of bytes to clear */
	void			*saveBuf,					/**<	a save area at least the size of the sector */
	UInt32			 saveBytes					/**<	save area size in bytes */
	 ) ;

/**	Read a sector.  The sector contents are read into the 'dataBuf' buffer.  The start and end 
 *	address do not have to start/end on sector boundaries, but no sector boundary may be crossed.  
 *	pgmAddr, dataBuf and dataBytes be aligned on even or odd byte boundaries, independent of FlashSctrAddr_t..
 *	 
 */
Result_t Flash_ReadSector( 
	FlashDev_t		*dev, 						/**<	the device */
	void			*pgmAddr,					/**<	the program address */
	void			*dataBuf,					/**<	area to receive the data */
	UInt32			 dataBytes					/**<	number of bytes to read */
	 ) ;

#endif // _WIN32

#endif
